using System;
using System.Collections;
using System.Data;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.ExceptionManagement;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>John Vrooland</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>9/9/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// An Invoice is an in-progress outgoing Shipment. 
	/// Until the shipment "leaves the building" the Invoice can be 
	/// appended to and deleted from. The business rules are somewhat 
	/// different than those of Shipment. Should perhaps be named OutgoingShipment.
	/// </summary>

	#endregion
	[Serializable]
	public class OutgoingShipmentCollection: BaseShipment, ICollection, IRecordStatusCode
	{
		/// <summary> 
		/// List of required columns in a rowset necessary to populate Invoice fields.
		/// </summary>
		public const string INVOICECOLUMNS = "ShipmentGuid,ShipmentDirection,ShipmentDate,InvoiceNumber,DivisionCode,LastUpdateDate," + "LastUpdateUser,ShipmentTemperature,ShipmentTypeCode,RowVersion,FacilityName,AddressCity," + "StateCode,LocalSupplierGuid,FdaRegistrationNumber,PhoneNumber";

		/// <summary>
		/// UnitStatus validation types
		/// </summary>
		public enum ValidStatus
		{
			/// <summary>
			/// Unit is not valid for shipment.
			/// </summary>
			No,
			/// <summary>
			/// Unit is valid for shipment
			/// </summary>
			Yes,
			/// <summary>
			/// Unit is not valid unless user approves
			/// </summary>
			Unknown
		} ;

		// 
		private LocalSupplier _localSupplier;
		private bool _readyToSave;
		private bool _readyToConfirm;
		private bool _readyToPrint;
		private bool _canceled;
		private bool _saved;
		private bool _confirmed;
		private bool _inErrorState;

		//
		//
		private DataTable _unitsInInvoice;
		private Hashtable _lineItems = new Hashtable();
		private RecordStatusCode _recordStatusCode;

		///
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		/// <summary>
		/// Private parameter-less constructor. Prevents instantiating 
		/// an empty invoice from outside.
		/// </summary>
		protected OutgoingShipmentCollection()
			: base()
		{
			//
			//
		}

		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2552"> 
		///		<ExpectedInput>Shipment GUID</ExpectedInput>
		///		<ExpectedOutput>OutgoingShipmentCollection object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2553"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="shipmentGuid"></param>
		public OutgoingShipmentCollection(Guid shipmentGuid) : base(shipmentGuid)
		{
			//
			this._localSupplier = new LocalSupplier(this.LocalSupplierGuid, true);
			//
			CheckIfConfirmed();
			//
			this.GetUnitsInInvoice(ShipmentGuid);
			//
			this._readyToSave = false;
			this._readyToConfirm = this.Count > 0 && !this._confirmed;
			this._readyToPrint = this.Count > 0;
			//this.InvoiceChanged += new EventHandler(this.OnInvoiceChanged);
		}

		//
		private void CheckIfConfirmed()
		{
			if (this.ShipmentDate > DateTime.MinValue)
			{
				this._confirmed = true;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2561"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2562"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ChangedHandler
		/// </summary>
		public void ChangedHandler()
		{
			this.ReadyToSave = true;
			this.OnReadyToSaveChanged();
		}

		#region Invoice Factories; static methods returning Invoice instance

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1090"> 
		///		<ExpectedInput>Valid invoice number, collecton facility GUID, shipment date, shipment type, division code, user ID</ExpectedInput>
		///		<ExpectedOutput>Valid invoice object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2491"> 
		///		<ExpectedInput>Invalid invoice number, collecton facility GUID, shipment date, shipment type, division code, user ID</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create an Outgoing Shipment Invoice with required fields.
		/// </summary>
		/// <param name="invoiceNumber"></param>
		/// <param name="localSupplierGuid"></param>
		/// <param name="shipmentDate"></param>
		/// <param name="shipmentType"></param>
		/// <param name="ntUserId"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <returns>An Invoice instance.</returns>
		public static OutgoingShipmentCollection CreateOutgoingInvoice(string invoiceNumber, Guid localSupplierGuid, DateTime shipmentDate, ShipmentType shipmentType, string ntUserId, UpdateFunction lastUpdateFunctionId)
		{
			if (invoiceNumber == null || invoiceNumber.Length > 14)
			{
				throw new ArgumentException("Invalid Argument:", "invoiceNumber");
			}

			if (localSupplierGuid == Guid.Empty)
			{
				throw new ArgumentException("Invalid Argument:", "LocalSupplierGuid");
			}

			if (shipmentDate.CompareTo(new DateTime(2001, 1, 1)) < 0 || shipmentDate.CompareTo(new DateTime(2045, 12, 31)) > 0)
			{
				throw new ArgumentException("Invalid Argument:", "shipmentDate");
			}

			if (shipmentType == ShipmentType.Unknown)
			{
				throw new ArgumentException("Invalid Argument:", "shipmentType");
			}

			if (ntUserId == null || ntUserId.Length > 30)
			{
				throw new ArgumentException("Invalid Argument:", "ntUserId");
			}

			if (lastUpdateFunctionId == UpdateFunction.Invalid)
			{
				throw new ArgumentException("Invalid Argument:", "lastUpdateFunctionId");
			}

			try
			{
				Guid shipmentGuid = Guid.NewGuid();
				//
				DAL.OutgoingShipment.CreateOutgoingInvoice( shipmentGuid, localSupplierGuid, Convert.ToChar(shipmentType), invoiceNumber, ntUserId, lastUpdateFunctionId );
				//
				return new OutgoingShipmentCollection( shipmentGuid );
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw new BusinessObjectException(StrRes.SysErrMsg.Common.GeneralError(Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString, ex);
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1092"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>Valid invoice</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2492"> 
		///		<ExpectedInput>Invalid GUID</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets an outgoing invoice where the guid is known.
		/// </summary>
		/// <param name="shipmentGuid"></param>
		/// <returns></returns>
		public static OutgoingShipmentCollection GetOutgoingInvoiceByGuid(Guid shipmentGuid)
		{
			OutgoingShipmentCollection invoice;

			DataTable dt = DAL.OutgoingShipment.GetOutgoingInvoiceByGuid(shipmentGuid).Tables[0];

			if (dt.Rows.Count < 1)
			{
				string errorMessage = StrRes.SysErrMsg.Common.NoRecordsFoundException("Invoice").ResString;
				throw new InvoiceException(errorMessage);
			}
			//
			invoice = new OutgoingShipmentCollection((Guid)dt.Rows[0][VbecsTables.Shipment.ShipmentGuid]);

			return invoice;
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1094"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid invoice</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2494"> 
		///		<ExpectedInput>Invalid data row</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return an invoice instance based on the contents of the DataRow provided.
		/// </summary>
		/// <param name="dr">DataRow containing Invoice columns</param>
		/// <returns></returns>
		public static OutgoingShipmentCollection GetInvoiceFromDataRow(DataRow dr)
		{
			OutgoingShipmentCollection invoice;

			//
			if (Utility.VerifyMinColumnsInDataTable(INVOICECOLUMNS.Split(','), dr.Table))
			{
				//
				bool x = dr.IsNull(VbecsTables.Shipment.ShipmentDate);
				Guid shipmentGuid = dr.IsNull(VbecsTables.Shipment.ShipmentGuid) ? Guid.Empty : (Guid)dr[VbecsTables.Shipment.ShipmentGuid];
				//
				invoice = new OutgoingShipmentCollection(shipmentGuid);
				//
				return invoice;
			}
			else
			{
				throw new InvoiceException(StrRes.SysErrMsg.Common.InsufficientColumns("invoice data row").ResString);
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="369"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Valid invoice array</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2495"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return an array of Invoices for this division, currently unused.
		/// </summary>
		/// <returns></returns>
		public static OutgoingShipmentCollection[] GetRecentInvoicesForDivisionArray()
		{
			try
			{
				DataTable dt = OutgoingShipmentCollection.GetRecentInvoicesForDivision();
				OutgoingShipmentCollection[] invoices = new OutgoingShipmentCollection[dt.Rows.Count + 1];
				int ndx = 0;

				foreach(DataRow dr in dt.Rows)
				{
					invoices[ndx++] = OutgoingShipmentCollection.GetInvoiceFromDataRow(dr);
				}
				return invoices;
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				string errorMessage = StrRes.SysErrMsg.Common.GeneralError(Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString;
				throw new InvoiceException(string.Concat(errorMessage, ": ", ex.Message), ex);
			}
		}

		// Other static methods
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2498"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>Valid data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2499"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return a list of invoices for this division, for selection.
		/// </summary>
		/// <returns>DataTable containing invoices</returns>
		public static DataTable GetRecentInvoicesForDivision()
		{
			try
			{
				DataTable dt = DAL.OutgoingShipment.GetRecentInvoicesForDivision();
				return dt;
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				string errorMessage = StrRes.SysErrMsg.Common.GeneralError(Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString;
				throw new InvoiceException(string.Concat(errorMessage, ": ", ex.Message), ex);
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="922"> 
		///		<ExpectedInput>Valid shipment type code</ExpectedInput>
		///		<ExpectedOutput>Valid Common.ShipmentType</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2500"> 
		///		<ExpectedInput>Invalid shipment type code</ExpectedInput>
		///		<ExpectedOutput>Common.ShipmentType.Unknown</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the shipment type of this code.
		/// </summary>
		/// <param name="shipmentTypeCode"></param>
		/// <returns></returns>
		public static ShipmentType GetShipmentTypeFromChar(char shipmentTypeCode)
		{
			ShipmentType shipmentType;

			//
			switch (shipmentTypeCode)
			{
				case ('1'):
					{
						shipmentType = ShipmentType.ReturnToSupplier;
						break;
					}

				case ('2'):
					{
						shipmentType = ShipmentType.SendToOtherVA;
						break;
					}

				case ('3'):
					{
						shipmentType = ShipmentType.SendToOtherFacility;
						break;
					}
				default:
					{
						shipmentType = ShipmentType.Unknown;
						break;
					}
			}
			return shipmentType;
			//
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2501"> 
		///		<ExpectedInput>Valid division code</ExpectedInput>
		///		<ExpectedOutput>More than one row returned</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2502"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieve a set of units appropriate for
		/// outgoing shipment.
		/// </summary>
		/// <returns>DataTable containg the units</returns>
		public static DataTable GetEligibleUnitsForDivision()
		{
			try
			{
				return DAL.OutgoingShipment.GetEligibleUnitsForDivision();
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				string errorMessage = StrRes.SysErrMsg.Common.GeneralError(Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString;
				throw new InvoiceException(string.Concat(errorMessage, ": ", ex.Message), ex);
			}
		}

		//

		#endregion

		#region Other Static Methods

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4057"> 
		///		<ExpectedInput>Valid unit status code</ExpectedInput>
		///		<ExpectedOutput>Valid status equal to yes</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4058"> 
		///		<ExpectedInput>Invalid unit status code</ExpectedInput>
		///		<ExpectedOutput>Valid status equal to no</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Determine whether or not unit is eligible for shipment
		/// or user should decide.
		/// </summary>
		/// <param name="unitStatusCode"></param>
		/// <returns></returns>
		public static ValidStatus EligibleUnitStatus(UnitStatusCode unitStatusCode)
		{
			Debug.WriteLine(unitStatusCode.ToString());

			switch (unitStatusCode)
			{
				case UnitStatusCode.Available:
					{
						return ValidStatus.Yes;
					}

				case UnitStatusCode.Limited:
					{
						return ValidStatus.Yes;
					}

				case UnitStatusCode.Assigned:
					{
						return ValidStatus.Yes;
					}

				case UnitStatusCode.Discarded:
					{
						return ValidStatus.No;
					}

				case UnitStatusCode.Issued:
					{
						return ValidStatus.No;
					}

				case UnitStatusCode.Transfused:
					{
						return ValidStatus.No;
					}

				case UnitStatusCode.Transferred:
					{
						return ValidStatus.No;
					}

				case UnitStatusCode.Modified:
					{
						return ValidStatus.No;
					}

				case UnitStatusCode.CrossmatchedAssigned:
					{
						return ValidStatus.Yes;
					}
				default:
					{
						return ValidStatus.Unknown;
					}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6854"> 
		///		<ExpectedInput>Valid BloodUnitGUid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6855"> 
		///		<ExpectedInput>Invalid BloodUnitguid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOutgoingBloodUnitShipment
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static DataTable GetOutgoingBloodUnitShipment(Guid bloodUnitGuid)
		{
			return DAL.Shipment.GetOutgoingBloodUnitShipment(bloodUnitGuid);
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/12/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7547"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7548"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements BR_11.04
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public static bool IsBloodUnitOnPendingOutgoingShipment(Guid bloodUnitGuid)
		{
			DataTable dtShipments = DAL.Shipment.GetOutgoingBloodUnitShipment(bloodUnitGuid);
			//
			foreach(DataRow dr in dtShipments.Rows)
			{
				if (dr.IsNull(TABLE.Shipment.ShipmentDate))
				{
					return true;
				}
			}
			//
			return false;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1510"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2506"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Get the text that appears above the signature on the printed invoice.
		/// </summary>
		/// <returns>DataRow containing the text.</returns>
		public static DataRow GetInvoiceText()
		{
			//
			DataTable dt;

			//
			try
			{
				dt = DAL.OutgoingShipment.GetInvoiceText();

				if (dt.Rows.Count > 0)
				{
					return dt.Rows[0];
				}
				else
				{
					string errorMessage = StrRes.SysErrMsg.Common.NoRecordsFoundException("Invoice text").ResString;
					throw new NoRecordsFoundException(errorMessage);
				}
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				return null;
			}
			//
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/23/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1511"> 
		///		<ExpectedInput>Valid divisionCode, vbecsUserID string, and invoice text string</ExpectedInput>
		///		<ExpectedOutput>True</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2507"> 
		///		<ExpectedInput>Invalid vbecsUserID string, and invoice text string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Update the printed invoice text that appears above the signature.
		/// </summary>
		/// <param name="vbecsUser">The user's NT Login</param>
		/// <param name="invoiceText">The new text.</param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="rowversion">timestamp for the division record</param>
		/// <returns></returns>
		public static bool UpdateInvoiceText(string vbecsUser, string invoiceText, UpdateFunction lastUpdateFunctionId, byte[] rowversion)
		{
			if (vbecsUser == null || vbecsUser.Length > 30)
			{
				throw new ArgumentException("Invalid Argument:", "vbecsUser");
			}

			if (invoiceText == null || invoiceText.Length > 1000)
			{
				throw new ArgumentException("Invalid Argument:", "InvoiceText");
			}

			if (lastUpdateFunctionId == UpdateFunction.Invalid)
			{
				throw new ArgumentException("Invalid Argument:", "lastUpdateFunctionId");
			}

			// 11/30/2004 - Added row version to the UpdateInvoiceText method of the
			// DAL.OutgoingShipment class.
			try
			{
				return DAL.OutgoingShipment.UpdateInvoiceText(vbecsUser, invoiceText, lastUpdateFunctionId, rowversion);
			}
			catch (DataAccessLayerException ex)
			{
				ExceptionManager.Publish(ex);
				string errorMessage = StrRes.SysErrMsg.Common.DataNotSaved("Invoice text").ResString;
				throw new InvoiceException(errorMessage, ex);
			}
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>05/17/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7305"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7306"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets count of shipment invoices that meet the 'UCK_Shipment' constraint.
		/// </summary>
		/// <param name="localSupplierGuid">Local Supplier Guid</param>
		/// <param name="invoiceNumber">Invoice Number</param>
		/// <returns>count (int)</returns>
		public static int GetShipmentInvoiceCount( Guid localSupplierGuid, string invoiceNumber )
		{
			return DAL.OutgoingShipment.GetShipmentInvoiceCount( localSupplierGuid, invoiceNumber );
		}

		//

		#endregion

		#region Private instance methods

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///
		/// <summary>
		/// Retrieve the list of units assigned to this invoice. From a previously saved
		/// Invoice.
		/// </summary>
		/// <param name="shipmentGuid"></param>
		private void GetUnitsInInvoice(Guid shipmentGuid)
		{
			_unitsInInvoice = DAL.OutgoingShipment.GetUnitsInInvoice(shipmentGuid);
			//
			_unitsInInvoice.Columns[VbecsTables.BloodUnit.BloodUnitGuid].Unique = true;

			//
			foreach(DataRow dr in _unitsInInvoice.Rows)
			{
				try
				{
					AddLineItem(dr);
				}
				catch (Exception ex)
				{
					ExceptionManager.Publish(ex);
					Utility.MessageBeep();
				}
			}
			_unitsInInvoice.AcceptChanges();
			//
			_unitsInInvoice.RowChanged += new DataRowChangeEventHandler(this.OnRowChanged);
			_unitsInInvoice.RowDeleted += new DataRowChangeEventHandler(this.OnRowDeleted);
			//
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2003</CreationDate>
		///
		/// <summary>
		/// Add an existing line item to this invoice from db.
		/// </summary>
		/// <param name="dr"></param>
		private void AddLineItem(DataRow dr)
		{
			if (this.Confirmed)
			{
				this.AddUnitToInvoice(LineItem.GetLineItem(((Guid)dr[VbecsTables.BloodUnit.BloodUnitGuid])));
			}
			else
			{
				this.AddUnitToInvoice(LineItem.GetLineItemForUpdate(((Guid)dr[VbecsTables.BloodUnit.BloodUnitGuid])));
			}
		}

		#endregion

		#region Protected or internal methods

		//
		/// <summary>
		/// Handler triggered on row change, indicates a save is necessary.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		protected void OnRowChanged(object sender, DataRowChangeEventArgs eventArgs)
		{
			if (eventArgs.Action != DataRowAction.Nothing && !_confirmed)
			{
				this.ReadyToSave = true;
				this.ReadyToConfirm = false;
				Debug.WriteLine("    RowChanged: Action = " + eventArgs.Action + ", BloodUnitGuid = " + eventArgs.Row[VbecsTables.BloodUnit.BloodUnitGuid]);
			}
		}

		/// <summary>
		/// Handler triggered on row deletion. A save is necessary.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		protected void OnRowDeleted(object sender, DataRowChangeEventArgs eventArgs)
		{
			if (eventArgs.Action != DataRowAction.Nothing && !_confirmed)
			{
				this.ReadyToSave = true;
				this.ReadyToConfirm = false;
				Debug.WriteLine("    RowChanged: Action = " + eventArgs.Action + ", BloodUnitGuid = " + eventArgs.Row[VbecsTables.BloodUnit.BloodUnitGuid]);
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// ReadyToSaveChanged
		/// </summary>
		protected void OnReadyToSaveChanged()
		{
			if ((ReadyToSaveChanged != null))
			{
				ReadyToSaveChanged(this, new ReadyToSaveChangedEventArgs(_readyToSave));
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// ReadyToConfirmChanged
		/// </summary>
		protected void OnReadyToConfirmChanged()
		{
			if ((ReadyToConfirmChanged != null))
			{
				ReadyToConfirmChanged(this, new ReadyToConfirmChangedEventArgs(_readyToConfirm));
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// ReadyToPrintChanged
		/// </summary>
		protected void OnReadyToPrintChanged()
		{
			if ((ReadyToPrintChanged != null))
			{
				ReadyToPrintChanged(this, new ReadyToPrintChangedEventArgs(_readyToPrint));
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// CanceledChanged
		/// </summary>
		protected void OnCanceledChanged()
		{
			if ((CanceledChanged != null))
			{
				CanceledChanged(this, new CanceledChangedEventArgs(_canceled));
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// SavedChanged
		/// </summary>
		protected void OnSavedChanged()
		{
			if ((SavedChanged != null))
			{
				SavedChanged(this, new SavedChangedEventArgs(_saved));
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// ConfirmedChanged
		/// </summary>
		protected void OnConfirmedChanged()
		{
			if ((ConfirmedChanged != null))
			{
				ConfirmedChanged(this, new ConfirmedChangedEventArgs(_confirmed));
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// InErrorStateChanged
		/// </summary>
		protected void OnInErrorStateChanged()
		{
			if ((InErrorStateChanged != null))
			{
				// fire event
				InErrorStateChanged(this, System.EventArgs.Empty);
			}
		}

		/// <summary>
		/// OnInvoiceChanged
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		protected void OnInvoiceChanged(object sender, EventArgs e)
		{
			this.ReadyToSave = true;
		}

		#endregion

		#region Public instance methods

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2563"> 
		///		<ExpectedInput>Valid parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2564"> 
		///		<ExpectedInput>Invalid parameters</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SaveChangesToInvoiceItems
		/// </summary>
		/// <param name="errorList"></param>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public void SaveChangesToInvoiceItems(ArrayList errorList, UpdateFunction lastUpdateFunctionId)
		{
			int nCount = 0;
			// Added on 3/1/2005 RDH
			// The following if block is used to set any existing records in the blood unit 
			// shipment table to inactive. This is because of the way the method was originally
			// written. Initially, it iterates through the collection of line items and adds
			// the items individually to the blood unit shipment table regardless of whether
			// the line item already exists. Doing this causes a uniqueness error. The following
			// if block prevents the uniqueness error from happening.
			if(this._unitsInInvoice.GetChanges()!=null && this._unitsInInvoice.GetChanges().Rows.Count>0)
			{
				DataTable dtRemoveShipment=new DataTable(TABLE.Shipment.TableName);
	
				dtRemoveShipment.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));
				dtRemoveShipment.Columns.Add(TABLE.Shipment.LastUpdateUser, typeof(string));
				dtRemoveShipment.Columns.Add(TABLE.Shipment.RowVersion, typeof(byte[]));
				dtRemoveShipment.Columns.Add(TABLE.Shipment.LastUpdateFunctionId, typeof(int));
				dtRemoveShipment.Columns.Add(TABLE.Shipment.LastUpdateDate, typeof(DateTime));
				nCount = this._unitsInInvoice.Rows.Count;
				for(int i=0; i<nCount; i++)
				{

					if(this._unitsInInvoice.Rows[i].RowState!=DataRowState.Added)
					{
						DataRow dtRow=dtRemoveShipment.NewRow();

						dtRow[TABLE.Shipment.ShipmentGuid]=this.ShipmentGuid;
						dtRow[TABLE.Shipment.RowVersion]=(byte[])this._unitsInInvoice.Rows[i][TABLE.Shipment.RowVersion]; 
						dtRow[TABLE.Shipment.LastUpdateUser]=Common.LogonUser.LogonUserName;
						dtRow[TABLE.Shipment.LastUpdateFunctionId]=lastUpdateFunctionId;
						dtRow[TABLE.Shipment.LastUpdateDate]=BOL.VBECSDateTime.GetDivisionCurrentDateTime();

						dtRemoveShipment.Rows.Add(dtRow);
					}
				}	
				
				DAL.OutgoingShipment.RemoveAllUnitsFromInvoice(dtRemoveShipment.Clone(), dtRemoveShipment); 
			}

			// This is the same as the original method.
			foreach(LineItem li in _lineItems.Values)
			{
				li.BloodUnitStatus.UnitStatusDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime(Common.LogonUser.LogonUserDivisionCode);
				if(li.SaveOutgoingShipment(this, lastUpdateFunctionId))
					//if (li.Save(lastUpdateFunctionId))
				{
					//
					Debug.WriteLine("Saved :" + li.LabelUnitId);
					//li.Refresh();
				}
				else
				{
					errorList.Add(li.EyeReadableUnitId);
				}
			}
			
			this.ReadyToSave = false;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="967"> 
		///		<ExpectedInput>Valid user ID</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2510"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Remove all unit DataRows from the invoice.
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public void RemoveAllUnitsFromThisInvoice(UpdateFunction lastUpdateFunctionId)
		{
			int nCount = 0;
			DataTable dtRemoveShipment=new DataTable(TABLE.Shipment.TableName);
			
			dtRemoveShipment.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));
			dtRemoveShipment.Columns.Add(TABLE.Shipment.RowVersion, typeof(byte[]));
			dtRemoveShipment.Columns.Add(TABLE.Shipment.LastUpdateFunctionId, typeof(int));
			dtRemoveShipment.Columns.Add(TABLE.Shipment.LastUpdateUser, typeof(string));
			
			DataRow dtRow=dtRemoveShipment.NewRow();

			dtRow[TABLE.Shipment.ShipmentGuid]=this.ShipmentGuid;
			dtRow[TABLE.Shipment.RowVersion]=this.RowVersion; 
			dtRow[TABLE.Shipment.LastUpdateUser]=Common.LogonUser.LogonUserName;
			dtRow[TABLE.Shipment.LastUpdateFunctionId]=lastUpdateFunctionId;

			dtRemoveShipment.Rows.Add(dtRow);

			DataTable dtRemoveBloodUnits=dtRemoveShipment.Clone();
			nCount = this._unitsInInvoice.Rows.Count;
			for(int i=0; i<nCount; i++)
			{
				if(this._unitsInInvoice.Rows[i].RowState!=DataRowState.Added)
				{
					dtRow=dtRemoveBloodUnits.NewRow();

					dtRow[TABLE.Shipment.ShipmentGuid]=this.ShipmentGuid;
					dtRow[TABLE.Shipment.RowVersion]=(byte[])this._unitsInInvoice.Rows[i][TABLE.Shipment.RowVersion]; 
					dtRow[TABLE.Shipment.LastUpdateUser]=Common.LogonUser.LogonUserName;
					dtRow[TABLE.Shipment.LastUpdateFunctionId]=lastUpdateFunctionId;
			
					dtRemoveBloodUnits.Rows.Add(dtRow);
				}
			}

			// Commented out on 12/16/2004 RDH updated the DAL method to accept 
			// the table of blood units contained within the shipment being
			// canceled.
			// DAL.OutgoingShipment.RemoveAllUnitsFromInvoice(this.ShipmentGuid, lastUpdateFunctionId);
			if(dtRemoveBloodUnits.Rows.Count>0)
				DAL.OutgoingShipment.RemoveAllUnitsFromInvoice(dtRemoveShipment,dtRemoveBloodUnits);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2003</CreationDate>
		///
		/// <summary>
		/// Remove all the line item rows from this invoice.
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		private void ClearInvoice(UpdateFunction lastUpdateFunctionId)
		{
			Array liList = Array.CreateInstance(typeof(LineItem), _lineItems.Count + 1);
			//
			_lineItems.Values.CopyTo(liList, 0);

			//
			foreach(LineItem li in liList)
			{
				//dr.Delete();
				RemoveUnitFromInvoice(li);
				//li.RemoveFromOutgoingShipment();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="968"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2511"> 
		///		<ExpectedInput>Invalid lineItem</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Remove a single row from the invoice.
		/// </summary>
		/// <param name="li"></param>
		public void RemoveUnitFromInvoice(LineItem li)
		{
			try
			{
				if (li != null)
				{
					RemoveFromInvoice(li);
				}
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				string errorMessage = StrRes.SysErrMsg.Common.GeneralError(Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString;
				throw new InvoiceException(string.Concat(errorMessage, ": ", ex.Message), ex);
			}
		}

		private void RemoveFromInvoice(LineItem li)
		{
			// Added on 3/1/2005 by RDH
			// Removed this method call because it was hitting the database even if the
			// user did not press the save button or if the item does not actually exist 
			// in the db, i.e. it is a new invoice or unit of the invoice not yet saved.
			//li.RemoveFromOutgoingShipment(lastUpdateFunctionId);
			//
			_lineItems.Remove(li.BloodUnitGuid.ToString());
			//
			DataRow[] dr = _unitsInInvoice.Select(VbecsTables.BloodUnit.BloodUnitGuid + " = '" + li.BloodUnitGuid.ToString() + "'");

			if (dr.Length > 0)
			{
				// Instead of deleting the record, set the record status code to inactive.
				// Later we will have access to the row version and record status code
				// when save the line item.
				dr[0][TABLE.Shipment.RecordStatusCode]=Common.Utility.GetRecordStatusCodeCharFromEnum(Common.RecordStatusCode.Inactive);
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="969"> 
		///		<ExpectedInput>Valid LineItem</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2512"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Add a unit row to the invoice.
		/// Implements BR_11.14
		/// </summary>
		/// <param name="lineItem"></param>
		public void AddUnitToInvoice(LineItem lineItem)
		{
			if (!_lineItems.ContainsKey(lineItem.BloodUnitGuid.ToString()))
			{
				_lineItems.Add(lineItem.BloodUnitGuid.ToString(), lineItem);
				//
				lineItem.OutgoingShipmentGuid = this.ShipmentGuid;
				//
				// BR_11.14: Default return credit if the units are being shipped back to original location
				if ( this.ShipmentType == Common.ShipmentType.ReturnToSupplier )
				{
					decimal defaultReturnCredit = lineItem.BloodUnitFinance.CalculateReturnCreditAmount( this.LocalSupplierGuid, lineItem.BloodProductGuid );
					//
					ModifyReturnCredit( lineItem, defaultReturnCredit );
				}
				//
				//
				if (_unitsInInvoice.Select(VbecsTables.BloodUnit.BloodUnitGuid + " = '" + lineItem.BloodUnitGuid.ToString() + "'").Length == 0)
				{
					DataRow dr = _unitsInInvoice.NewRow();
					dr[VbecsTables.BloodUnit.BloodUnitGuid] = lineItem.BloodUnitGuid;
					dr[VbecsTables.Shipment.RecordStatusCode]=Common.Utility.GetRecordStatusCodeCharFromEnum(lineItem.RecordStatusCode);
					_unitsInInvoice.Rows.Add(dr);
					OnInvoiceChanged(this, EventArgs.Empty);
				}
				//
			}
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2565"> 
		///		<ExpectedInput>LineItem, decimal</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2566"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ModifyReturnCredit
		/// </summary>
		/// <param name="li"></param>
		/// <param name="returnCreditAmount"></param>
		public void ModifyReturnCredit(LineItem li, decimal returnCreditAmount)
		{
			li.ReturnCreditAmount = returnCreditAmount;
			OnInvoiceChanged(this, EventArgs.Empty);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="970"> 
		///		<ExpectedInput>Valid user ID and shipdate</ExpectedInput>
		///		<ExpectedOutput>Hashtable with confirmed statuses</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2513"> 
		///		<ExpectedInput>Invalid user ID and shipdate</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// User has indicated shipment confirmation, set appropriate statuses for each
		/// unit in the Invoice.
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		/// <param name="shipDate">Date</param>
		/// <param name="workloadProcessId">Workload Process ID</param>
		public bool ConfirmShipment(
						DateTime shipDate, 
						UpdateFunction lastUpdateFunctionId, 
						WorkloadProcessID workloadProcessId
						)
		{
			if (shipDate == DateTime.MinValue || shipDate.CompareTo(new DateTime(2001, 1, 1)) < 0 || shipDate.CompareTo(new DateTime(2045, 12, 31)) > 0)
			{
				throw new ArgumentException("Invalid Argument:", "shipDate");
			}
			this.ShipmentDate = shipDate;

			if (!_readyToConfirm)
			{
				string errorMessage = StrRes.SysErrMsg.Common.ConfirmShipmentError().ResString;
				throw new InvoiceException(errorMessage);
			}

			try
			{
				bool goodToGo = this.ProcessConfirmShipment(lastUpdateFunctionId, workloadProcessId);

				//
				if (goodToGo)
				{
					this._confirmed = true;
					ReadyToConfirm = false;
					ReadyToSave = false;
					ReadyToPrint = true;
					return true;
				}
			}
			catch (Exception ex)
			{
				Debug.WriteLine(ex.Message + "\n" + ex.StackTrace);
				ExceptionManager.Publish(ex);
				string errorMessage = StrRes.SysErrMsg.Common.GeneralError(Utility.ParseStringwithSpaces(MethodBase.GetCurrentMethod().Name)).ResString;
				throw new InvoiceException(string.Concat(errorMessage, ": ", ex.Message), ex);
			}
			return false;
		}

		//
		private bool ProcessConfirmShipment(UpdateFunction lastUpdateFunctionId, WorkloadProcessID workloadProcessId)
		{
			DataTable dtBloodUnitsInShipment = DAL.OutgoingShipment.GetUnitsInInvoice(this.ShipmentGuid);
			Guid[] bloodUnitGuids = new Guid[dtBloodUnitsInShipment.Rows.Count];
			int nCtr = 0;

			foreach(DataRow dtShipmentRow in dtBloodUnitsInShipment.Rows)
			{
				bloodUnitGuids[nCtr] = (Guid)dtShipmentRow[VbecsTables.BloodUnit.BloodUnitGuid];
				nCtr++;
			}
			DataTable dtWorkload = WorkloadEvent.GenerateWorkloadData(workloadProcessId, bloodUnitGuids, Common.WorkloadTransactionType.Unit, false);
			Common.Utility.AppendLastUpdateInformation(dtWorkload,lastUpdateFunctionId);

			// This is the shipment being confirmed.
			System.Data.DataTable dtShipment = new System.Data.DataTable();
			dtShipment.Columns.Add(TABLE.Shipment.ShipmentGuid,typeof(System.Guid));
			dtShipment.Columns.Add(TABLE.Shipment.ShipmentDate, typeof(System.DateTime));
			dtShipment.Columns.Add(TABLE.Shipment.LastUpdateUser, typeof(System.String));
			dtShipment.Columns.Add(TABLE.Shipment.LastUpdateFunctionId, typeof(System.Int32));
			dtShipment.Columns.Add(TABLE.Shipment.DivisionCode, typeof(System.String));
			dtShipment.Columns.Add(TABLE.Shipment.RowVersion, typeof(byte[]));

			System.Data.DataRow dtRow = dtShipment.NewRow();
			dtRow[TABLE.Shipment.ShipmentGuid] = this.ShipmentGuid;
			dtRow[TABLE.Shipment.ShipmentDate] = this.ShipmentDate;
			dtRow[TABLE.Shipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dtRow[TABLE.Shipment.LastUpdateFunctionId] = lastUpdateFunctionId;
			dtRow[TABLE.Shipment.DivisionCode] = LogonUser.LogonUserDivisionCode;
			dtRow[TABLE.Shipment.RowVersion] = this.RowVersion;
			dtShipment.Rows.Add(dtRow);

			// Units contained within the shipment that are confirmed.
			System.Data.DataTable dtBloodUnit = new System.Data.DataTable();
			dtBloodUnit.Columns.Add(TABLE.Shipment.ShipmentGuid, typeof(System.Guid));
			dtBloodUnit.Columns.Add(TABLE.Shipment.ShipmentDate, typeof(System.DateTime));
			dtBloodUnit.Columns.Add(TABLE.Shipment.LastUpdateUser, typeof(System.String));
			dtBloodUnit.Columns.Add(TABLE.Shipment.LastUpdateFunctionId, typeof(System.Int32));
			dtBloodUnit.Columns.Add(TABLE.BloodUnit.RowVersion, typeof(byte[]));						

			// Adding the units to the table
			foreach(LineItem item in this._lineItems.Values)
			{
				System.Data.DataRow unitRow = dtBloodUnit.NewRow();
				unitRow[TABLE.Shipment.ShipmentGuid] = this.ShipmentGuid;
				unitRow[TABLE.Shipment.ShipmentDate] = this.ShipmentDate;
				unitRow[TABLE.Shipment.LastUpdateUser] = Common.LogonUser.LogonUserName;
				unitRow[TABLE.Shipment.LastUpdateFunctionId] = lastUpdateFunctionId;
				unitRow[TABLE.BloodUnit.RowVersion]=item.RowVersion;
				dtBloodUnit.Rows.Add(unitRow);
			}

			DataTable dtBCEMessages = null;
			DataTable dtTempBCEMessages = null;

			//CR 2941
			//CR 2945
			//generate table with BCE_MESSAGE
			//The Unit is removed from assignment when it is shipped out

			Guid bloodUnitGuid = Guid.Empty;
			Guid patientGuid = Guid.Empty;
			for (int i=0; i< dtBloodUnitsInShipment.Rows.Count; i++)
			{
				bloodUnitGuid = (Guid)dtBloodUnitsInShipment.Rows[i][Common.VbecsTables.BloodUnit.BloodUnitGuid];

				//getting all patients assigned to a unit
				DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(bloodUnitGuid);

				for (int j=0; j< dtPatients.Rows.Count; j++)
				{
					patientGuid = (Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid];
					//CR 3015
					if (dtBCEMessages == null)
					{
						dtBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnitGuid, patientGuid, false);
					}
					else
					{
						dtTempBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnitGuid, patientGuid, false);
						if (dtTempBCEMessages.Rows.Count > 0)
							dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
					}
				}
			}
			//BCE_MESSAGE end

			// As multiple people are adding units to this shipment,
			// it's unknown whether "they're all here" so we have to mark
			// all units with the right OutgoingShipmentGuid as shipped.
			bool result =  DAL.OutgoingShipment.ConfirmShipment(dtWorkload, dtBloodUnit, dtShipment, dtBCEMessages);
			
			return result;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1148"> 
		///		<ExpectedInput>string array of column names, Valid data table</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2514"> 
		///		<ExpectedInput>string array of column names, Invalid data table</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Validate the DataTable provided, to ensure all required data is available 
		/// for processing
		/// </summary>
		/// <param name="columnList">string array containing the required column names.</param>
		/// <param name="dt">The DataTable to be checked</param>
		/// <returns></returns>
		public bool VerifyMinColumnsInDataTable(string columnList, DataTable dt)
		{
			return Utility.VerifyMinColumnsInDataTable(columnList.Split(','), dt);
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1150"> 
		///		<ExpectedInput>Valid data row</ExpectedInput>
		///		<ExpectedOutput>Valid data row</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="447"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return a DataRow populated from an Invoice. Not currently useful.
		/// </summary>
		/// <param name="dataRow">Unused</param>
		/// <returns></returns>
		public override DataRow LoadDataRowFromThis(DataRow dataRow)
		{
			return dataRow;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="972"> 
		///		<ExpectedInput>Valid user ID</ExpectedInput>
		///		<ExpectedOutput>Hashtable of changed rows</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2515"> 
		///		<ExpectedInput>Invalid user ID</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Cancels an invoice
		/// </summary>
		/// <param name="thisUser">string containing the user's NTLogin</param>
		/// <returns>Hashtable</returns>
		public Hashtable CancelThisInvoice(string thisUser)
		{
			if (thisUser == null || thisUser.Length > 30)
			{
				throw new ArgumentException("Invalid Argument:", "thisUser");
			}

			//
			try
			{
				Hashtable ht = new Hashtable();

				if (!_confirmed)
				{
					foreach(DataRow dr in _unitsInInvoice.Rows)
					{
						dr.Delete();
					}
					//TODO: Replace
					//ht = SaveChangesToInvoice(thisUser);
				}
				return ht;
			}
			catch (Exception ex)
			{
				ExceptionManager.Publish(ex);
				throw ex;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1151"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="446"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>None</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Create the Report data for this invoice in the user's application data directory.
		/// </summary>
		public void GetXmlReportDataSet()
		{
			DataSet ds1 = DAL.OutgoingShipment.GetReportDataSet(this.ShipmentGuid);
			//
			string xmlPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData) + @"\Invoice_" + this.InvoiceNumber.Trim() + "_Report.xml";
			//
			FileStream headerFileStream = new FileStream(xmlPath, FileMode.Create);
			//
			XmlTextWriter headerXmlTextWriter = new XmlTextWriter(headerFileStream, Encoding.ASCII);
			ds1.WriteXml(headerXmlTextWriter, XmlWriteMode.WriteSchema);
#if     DEBUG
			ds1.WriteXmlSchema("C:\\InvoiceReport.xsd");
#endif
			headerFileStream.Close();

		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2575"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2576"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetReportDataSet
		/// </summary>
		/// <returns></returns>
		public DataSet GetReportDataSet()
		{
			DataSet ds = DAL.OutgoingShipment.GetReportDataSet(this.ShipmentGuid);
			DataSet dsReport = new DataSet();
			DataTable dtShipper = new DataTable("Table");
			//
			dtShipper.Columns.Add(ARTIFICIAL.SenderFacilityName);
			dtShipper.Columns.Add(ARTIFICIAL.SenderAddressLine1);
			dtShipper.Columns.Add(ARTIFICIAL.SenderAddressLine2);
			dtShipper.Columns.Add(ARTIFICIAL.SenderCityStateZip);
			dtShipper.Columns.Add(TABLE.Shipment.InvoiceNumber);
			dtShipper.Columns.Add(TABLE.Shipment.ShipmentDate);
			dtShipper.Columns.Add("Facility_Name");
			dtShipper.Columns.Add(TABLE.Address.AddressLine1);
			dtShipper.Columns.Add(TABLE.Address.AddressLine2);
			dtShipper.Columns.Add(ARTIFICIAL.CityStateZip);
			dtShipper.Columns.Add(ARTIFICIAL.Phone);
			dtShipper.Columns.Add(ARTIFICIAL.Fax);
			dtShipper.Columns.Add(ARTIFICIAL.TemplateText);
			dtShipper.Columns.Add(ARTIFICIAL.ProcessedBy);
			//
			DataRow drShipper = dtShipper.NewRow();
			drShipper[ARTIFICIAL.SenderFacilityName] = ds.Tables[0].Rows[0][ARTIFICIAL.SenderFacilityName];
			drShipper[ARTIFICIAL.SenderAddressLine1] = ds.Tables[0].Rows[0][ARTIFICIAL.SenderAddressLine1];
			drShipper[ARTIFICIAL.SenderAddressLine2] = ds.Tables[0].Rows[0][ARTIFICIAL.SenderAddressLine2];
			drShipper[ARTIFICIAL.SenderCityStateZip] = ds.Tables[0].Rows[0][ARTIFICIAL.SenderCityStateZip];
			drShipper[TABLE.Shipment.InvoiceNumber] = ds.Tables[0].Rows[0][TABLE.Shipment.InvoiceNumber];
			if (ds.Tables[0].Rows[0].IsNull(TABLE.Shipment.ShipmentDate))
			{
				drShipper[TABLE.Shipment.ShipmentDate] = "NOT CONFIRMED";
			}
			else
			{
				drShipper[TABLE.Shipment.ShipmentDate] = Common.VBECSDateTime.FormatDateTimeString(ds.Tables[0].Rows[0][TABLE.Shipment.ShipmentDate]);
			}
			drShipper["Facility_Name"] = ds.Tables[0].Rows[0][TABLE.LocalSupplier.FacilityName];
			drShipper[TABLE.Address.AddressLine1] = ds.Tables[0].Rows[0][TABLE.Address.AddressLine1];
			drShipper[TABLE.Address.AddressLine2] = ds.Tables[0].Rows[0][TABLE.Address.AddressLine2];
			drShipper[ARTIFICIAL.CityStateZip] = ds.Tables[0].Rows[0][Common.DatabaseConstants.ArtificialColumnNames.CityStateZip];
			drShipper[ARTIFICIAL.Phone] = ds.Tables[0].Rows[0][TABLE.LocalSupplier.PhoneNumber];
			drShipper[ARTIFICIAL.Fax] = ds.Tables[0].Rows[0][TABLE.LocalSupplier.FaxNumber];
			drShipper[ARTIFICIAL.TemplateText] = ds.Tables[0].Rows[0][TABLE.VamcDivision.InvoiceTemplateText];
			drShipper[ARTIFICIAL.ProcessedBy] = new BOL.VbecsUser(ds.Tables[0].Rows[0][TABLE.BloodUnit.LastUpdateUser].ToString()).UserName;
			//
			dtShipper.Rows.Add(drShipper);
			dsReport.Tables.Add(dtShipper);
			// Now, get units
			DataTable dtUnits = new DataTable("Table1");
			//
			dtUnits.Columns.Add("Unit_ID");
			dtUnits.Columns.Add("ABORh");
			dtUnits.Columns.Add("Exp_Date");
			dtUnits.Columns.Add("BioHazardous");
			dtUnits.Columns.Add("Product_Name");
			dtUnits.Columns.Add("Tests");
			dtUnits.Columns.Add(TABLE.ExceptionPatientTest.PatientName);

			//
			foreach(DataRow dr in ds.Tables[1].Rows)
			{
				DataRow drUnits = dtUnits.NewRow();
				drUnits["Unit_ID"] = dr[TABLE.BloodUnit.EyeReadableUnitId];
				drUnits["ABORh"] = dr[Common.DatabaseConstants.ArtificialColumnNames.ABORh];
				drUnits["Exp_Date"] = Common.VBECSDateTime.FormatDateTimeString(dr[TABLE.BloodUnitMedia.UnitExpirationDate]);
				drUnits["BioHazardous"] = dr[Common.DatabaseConstants.ArtificialColumnNames.Biohazardous];
				drUnits["Product_Name"] = dr[TABLE.BloodProduct.ProductName];
				drUnits["Tests"] = dr[Common.DatabaseConstants.ArtificialColumnNames.Tests];
				drUnits[TABLE.ExceptionPatientTest.PatientName] = dr[Common.DatabaseConstants.ArtificialColumnNames.PatientDisplayName];
				//
				dtUnits.Rows.Add(drUnits);
			}
			dsReport.Tables.Add(dtUnits);
			return dsReport;
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="965"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="442"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invoice DataTable has suffered a concurrency problem and must be closed;
		/// </summary>
		public bool InErrorState
		{
			get
			{
				return this._inErrorState;
			}
			set
			{
				this._inErrorState = value;
				OnInErrorStateChanged();
			}
		}

		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2577"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2578"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Number of units contained in this Invoice
		/// </summary>
		public int Count
		{
			get
			{
				if (_lineItems == null)
				{
					return 0;
				}
				return _lineItems.Count;
			}
		}

				
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2614"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2615"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SyncRoot
		/// </summary>
		public object SyncRoot
		{
			get
			{
				return _lineItems.SyncRoot;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2753"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2757"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IsSynchronized
		/// </summary>
		public bool IsSynchronized
		{
			get
			{
				return _lineItems.IsSynchronized;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2758"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>IEnumerator</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2785"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetEnumerator
		/// </summary>
		/// <returns></returns>
		public IEnumerator GetEnumerator()
		{
			if (_lineItems == null)
			{
				_lineItems = new Hashtable();
			}
			return _lineItems.GetEnumerator();
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2786"> 
		///		<ExpectedInput>Array, Index</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2809"> 
		///		<ExpectedInput>Invalid index</ExpectedInput>
		///		<ExpectedOutput>OutOfIndexRange Exception</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// CopyTo
		/// </summary>
		/// <param name="arr"></param>
		/// <param name="n"></param>
		public void CopyTo(Array arr, int n)
		{
			_lineItems.CopyTo(arr, n);
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2810"> 
		///		<ExpectedInput>Item Guid</ExpectedInput>
		///		<ExpectedOutput>LineItem</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2844"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Exception</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// LineItem
		/// </summary>
		public LineItem this[Guid item]
		{
			get
			{
				return (LineItem)_lineItems[item.ToString()];
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2845"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>ICollection</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2875"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Values
		/// </summary>
		public ICollection Values
		{
			get
			{
				return _lineItems.Values;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2888"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2889"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// IsDirty
		/// </summary>
		public new bool IsDirty
		{
			get
			{
				foreach(LineItem li in _lineItems.Values)
				{
					if (li.IsDirty)
					{
						return true;
					}
				}
				return base.IsDirty;
			}
			set
			{
				base.IsDirty = value;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="370"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid shipment GUID</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="440"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The Guid of this Outgoing Invoice.
		/// </summary>
		public new Guid ShipmentGuid
		{
			get
			{
				return base.ShipmentGuid;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="371"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="439"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The direction of this shipment. False for outgoing.
		/// </summary>
		public new bool ShipmentDirection
		{
			get
			{
				return base.ShipmentDirection;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="372"> 
		///		<ExpectedInput>Valid Date</ExpectedInput>
		///		<ExpectedOutput>Valid Date</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="436"> 
		///		<ExpectedInput>Invalid Date</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The Date/Time of this shipment.
		/// Can apparently be any time in the past, 
		/// present or future.
		/// </summary>
		public new DateTime ShipmentDate
		{
			get
			{
				return base.ShipmentDate;
			}
			set
			{
				if (value != DateTime.MinValue && value.CompareTo(new DateTime(2001, 1, 1)) < 0 || value.CompareTo(new DateTime(2045, 12, 31)) > 0)
				{
					throw new ArgumentException("Invalid Argument:", "value");
				}
				base.ShipmentDate = value;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="373"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="435"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// The invoice number of this invoice. Can be entered or 
		/// automatically generated in the GUI.
		/// </summary>
		public new string InvoiceNumber
		{
			get
			{
				return base.InvoiceNumber;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="377"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid short</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="433"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// Get the shipment temperature. 
		/// </summary>
		public new short ShipmentTemperature
		{
			get
			{
				return base.ShipmentTemperature;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="923"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid Common.ShipmentType</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="429"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Shipment type; return to supplier, Other VA Hospital, Other Facility.
		/// </summary>
		public ShipmentType ShipmentType
		{
			get
			{
				return GetShipmentTypeFromChar(base.ShipmentTypeCode[0]);
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/9/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="382"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid GUID</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="428"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Return the Guid of this shipment's destination.
		/// </summary>
		public new Guid LocalSupplierGuid
		{
			get
			{
				return base.LocalSupplierGuid;
			}
		}

		//
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="921"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid local supplier</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="424"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The facility this shipment is destined for.
		/// </summary>
		public LocalSupplier LocalSupplier
		{
			get
			{
				return _localSupplier;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/20/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="920"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="421"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// The name of the ship-to facility.
		/// </summary>
		public string FacilityName
		{
			get
			{
				if (!(this._localSupplier == null))
				{
					return this._localSupplier.FacilityName;
				}
				else
				{
					return "";
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2516"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="419"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invoice has been modified, requiring a save.
		/// </summary>
		public bool ReadyToSave
		{
			get
			{
				return this._readyToSave;
			}
			set
			{
				this._readyToSave = value;
				this.ReadyToPrint = !this._readyToSave;
				this.ReadyToConfirm = !this.ReadyToSave && this._lineItems.Count > 0;
				OnReadyToSaveChanged();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2517"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="418"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invoice is confirmed. Only printing is allowed.
		/// </summary>
		public bool ReadyToPrint
		{
			get
			{
				return this._readyToPrint;
			}
			set
			{
				this._readyToPrint = value;
				OnReadyToPrintChanged();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2518"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="415"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invoice has been saved, valid units are present.
		/// </summary>
		public bool ReadyToConfirm
		{
			get
			{
				return this._readyToConfirm;
			}
			set
			{
				if (!this._confirmed)
				{
					this._readyToConfirm = value;
					OnReadyToConfirmChanged();
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2890"> 
		///		<ExpectedInput>Common.RecordStatusCode</ExpectedInput>
		///		<ExpectedOutput>Common.RecordStatusCode</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="2891"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// The status of the record
		/// </summary>
		public RecordStatusCode RecordStatusCode
		{
			get
			{
				return _recordStatusCode;
			}
			set
			{
				_recordStatusCode = value;
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2519"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="410"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invoice has been canceled, all units will be removed.
		/// </summary>
		public bool Canceled
		{
			get
			{
				return this._canceled;
			}
			set
			{
				this._canceled = value;
				OnCanceledChanged();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="2520"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="414"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Invoice has been saved
		/// </summary>
		public bool Saved
		{
			get
			{
				return this._saved;
			}
			set
			{
				this._saved = value;
				this.ReadyToConfirm = this._saved && (this._lineItems.Count > 0);
				OnSavedChanged();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/6/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2521"> 
		///		<ExpectedInput>Valid boolean</ExpectedInput>
		///		<ExpectedOutput>Valid boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="411"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Invoice has been confirmed
		/// </summary>
		public bool Confirmed
		{
			get
			{
				return this._confirmed;
			}
			set
			{
				this._confirmed = value;
				OnConfirmedChanged();
			}
		}

		#endregion

		#region event and delegate variables

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		public event System.EventHandler InErrorStateChanged;
		
		/// <summary>
		/// ReadyToSaveChanged EventHandler
		/// </summary>
		public event ReadyToSaveChangedEventHandler ReadyToSaveChanged;

		/// <summary>
		/// ReadyToConfirmChanged EventHandler
		/// </summary>
		public event ReadyToConfirmChangedEventHandler ReadyToConfirmChanged;

		/// <summary>
		/// ReadyToPrintChanged EventHandler
		/// </summary>
		public event ReadyToPrintChangedEventHandler ReadyToPrintChanged;

		/// <summary>
		/// CanceledChanged EventHandler
		/// </summary>
		public event CanceledChangedEventHandler CanceledChanged;

		/// <summary>
		/// SavedChanged EventHandler
		/// </summary>
		public event SavedChangedEventHandler SavedChanged;

		/// <summary>
		/// ConfirmedChanged EventHandler
		/// </summary>
		public event ConfirmedChangedEventHandler ConfirmedChanged;

		

		/// <summary>
		/// ReadyToSaveChangedEventHandler Delegate
		/// </summary>
		public delegate void ReadyToSaveChangedEventHandler(object sender, ReadyToSaveChangedEventArgs e);

		/// <summary>
		/// ReadyToConfirmChangedEventHandler Delegate
		/// </summary>
		public delegate void ReadyToConfirmChangedEventHandler(object sender, ReadyToConfirmChangedEventArgs e);

		/// <summary>
		/// ReadyToPrintChangedEventHandler Delegate
		/// </summary>
		public delegate void ReadyToPrintChangedEventHandler(object sender, ReadyToPrintChangedEventArgs e);

		/// <summary>
		/// CanceledChangedEventHandler Delegate
		/// </summary>
		public delegate void CanceledChangedEventHandler(object sender, CanceledChangedEventArgs e);

		/// <summary>
		/// SavedChangedEventHandler Delegate
		/// </summary>
		public delegate void SavedChangedEventHandler(object sender, SavedChangedEventArgs e);

		/// <summary>
		/// ConfirmedChangedEventHandler Delegate
		/// </summary>
		public delegate void ConfirmedChangedEventHandler(object sender, ConfirmedChangedEventArgs e);

		/// <summary>
		/// InErrorStateChangedEventHandler Delegate
		/// </summary>
		public delegate void InErrorStateChangedEventHandler(object sender, System.EventArgs e);

		//

		#endregion

		//
		//

		#region LineItem class

		/// <summary>
		/// Lineitem class
		/// </summary>
		public class LineItem: BloodUnit
		{
			/// <summary>
			/// _validForShipment
			/// </summary>
			private bool _validForShipment;

			/// <summary>
			/// _originalReturnCreditAmount
			/// </summary>
			private decimal _originalReturnCreditAmount;

			/// <summary>
			/// LineItem
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			private LineItem(Guid bloodUnitGuid)
				: base(bloodUnitGuid)
			{
				_validForShipment = (this.EligibleUnitStatus() == ValidStatus.Yes);
				_originalReturnCreditAmount = this.ReturnCreditAmount;
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6423"> 
			///		<ExpectedInput>Valid Guid</ExpectedInput>
			///		<ExpectedOutput>LineItem</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6424"> 
			///		<ExpectedInput>Invalid Guid</ExpectedInput>
			///		<ExpectedOutput>Exception</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// GetLineItemForUpdate
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			/// <returns></returns>
			public static LineItem GetLineItemForUpdate(Guid bloodUnitGuid)
			{
				//
				LineItem li;
				li = new LineItem(bloodUnitGuid);

				return li;
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6425"> 
			///		<ExpectedInput>valid BloodUnitGuid</ExpectedInput>
			///		<ExpectedOutput>LineItem</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6426"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// GetLineItem
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			/// <returns></returns>
			public static LineItem GetLineItem(Guid bloodUnitGuid)
			{
				//
				//LineItem li = new LineItem(bloodUnitGuid);
				//
				//return li;

				return new LineItem(bloodUnitGuid);
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6431"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6432"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// ValidForShipment
			/// </summary>
			public bool ValidForShipment
			{
				get
				{
					return _validForShipment;
				}
				set
				{
					if (this.EligibleUnitStatus() == ValidStatus.No && value == true)
					{
						throw new ArgumentException("UnitStatusCode " + this.BloodUnitStatus.UnitStatusCode + " is not valid for shipment.");
					}
					else
					{
						_validForShipment = value;
					}
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6433"> 
			///		<ExpectedInput>Guid</ExpectedInput>
			///		<ExpectedOutput>Guid</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6434"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// OutgoingShipmentGuid
			/// </summary>
			public Guid OutgoingShipmentGuid
			{
				get
				{
					return base.BloodUnitFinance.OutgoingShipmentGuid;
				}
				set
				{
					base.BloodUnitFinance.OutgoingShipmentGuid = value;
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6438"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>string</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6439"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// ABORh
			/// </summary>
			public string AboRh
			{
				get
				{
					return base.BloodUnitMedia.AboRh.AboRHText;
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6440"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6441"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// IsBiohazardous
			/// </summary>
			public bool IsBiohazardous
			{
				get
				{
					return base.BloodUnitStatus.BiohazardousWarningIndicator;
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6442"> 
			///		<ExpectedInput>Decimal</ExpectedInput>
			///		<ExpectedOutput>Decimal</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6443"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// ReturnCreditAmount
			/// </summary>
			public decimal ReturnCreditAmount
			{
				get
				{
					return base.BloodUnitFinance.ReturnCreditAmount;
				}
				set
				{
					base.BloodUnitFinance.ReturnCreditAmount = value;
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6444"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>string</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6445"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// PatientName
			/// </summary>
			public string PatientName
			{
				get
				{
					string fullNameString = (base.DirectedPatient.LastName + ", " + base.DirectedPatient.FirstName + " " + base.DirectedPatient.MiddleInitial);

					if (fullNameString.Trim() == ",")
					{
						return "";
					}
					else
					{
						return fullNameString.Trim();
					}
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6446"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6447"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// IsQuarantined
			/// </summary>
			public bool IsQuarantined
			{
				get
				{
					return base.BloodUnitStatus.QuarantineIndicator;
				}
			}

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/19/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6448"> 
			///		<ExpectedInput>None</ExpectedInput>
			///		<ExpectedOutput>ValidStatus code</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6449"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// EligibleUnitStatus
			/// </summary>
			/// <returns></returns>
			public ValidStatus EligibleUnitStatus()
			{
				switch (this.BloodUnitStatus.UnitStatusCode)
				{
					case UnitStatusCode.Available:
						{
							return ValidStatus.Yes;
						}

					case UnitStatusCode.Limited:
						{
							return ValidStatus.Yes;
						}

					case UnitStatusCode.Assigned:
						{
							return ValidStatus.Yes;
						}

					case UnitStatusCode.Discarded:
						{
							return ValidStatus.No;
						}

					case UnitStatusCode.Issued:
						{
							return ValidStatus.No;
						}

					case UnitStatusCode.Transfused:
						{
							return ValidStatus.No;
						}

					case UnitStatusCode.Transferred:
						{
							return ValidStatus.No;
						}

					case UnitStatusCode.Modified:
						{
							return ValidStatus.No;
						}

					case UnitStatusCode.CrossmatchedAssigned:
						{
							return ValidStatus.Unknown;
						}
					default:
						{
							return ValidStatus.Unknown;
						}
				}
			}
		}

		#endregion

		#region EventArgs Classes

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// Used to pass an indication that a invoice is in need of saving
		/// or not.
		/// </summary>
		public class ReadyToSaveChangedEventArgs: EventArgs
		{
			private bool _readyToSave;

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/7/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6874"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>ReadyToSaveChangedEventArgs</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6875"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Constructor for passing the Saved 
			/// indicator.
			/// </summary>
			/// <param name="readyToSave"></param>
			public ReadyToSaveChangedEventArgs(bool readyToSave)
			{
				_readyToSave = readyToSave;
			}

			///<Developers>
			///	<Developer>John Vrooland</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/21/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6876"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6877"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			///Is invoice ready to save
			/// </summary>
			public bool ReadyToSave
			{
				get
				{
					return _readyToSave;
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// A confirmed state change indicating
		/// whether a invoice is ready for shipment
		/// </summary>
		public class ReadyToConfirmChangedEventArgs: EventArgs
		{
			private bool _readyToConfirm;

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/7/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6878"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>ReadyToConfirmChangedEventArgs object</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6879"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Constructor for this thing
			/// </summary>
			/// <param name="readyToConfirm"></param>
			public ReadyToConfirmChangedEventArgs(bool readyToConfirm)
			{
				_readyToConfirm = readyToConfirm;
			}

			///<Developers>
			///	<Developer>John Vrooland</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/21/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6880"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6881"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// If invoice is ready to confirm
			/// </summary>
			public bool ReadyToConfirm
			{
				get
				{
					return _readyToConfirm;
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// Passes Invoice has been confirmed, print is now only option.
		/// </summary>
		public class ReadyToPrintChangedEventArgs: EventArgs
		{
			private bool _readyToPrint;


			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/7/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6882"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>ReadyToPrintChangedEventArgs object</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6883"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Constructor for this thing
			/// </summary>
			/// <param name="readyToPrint">bool indicator</param>
			public ReadyToPrintChangedEventArgs(bool readyToPrint)
			{
				_readyToPrint = readyToPrint;
			}


			///<Developers>
			///	<Developer>John Vrooland</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/21/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6884"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6885"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// If invoice is ready to print
			/// </summary>
			public bool ReadyToPrint
			{
				get
				{
					return _readyToPrint;
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// Indication of an invoice cancellation.
		/// </summary>
		public class CanceledChangedEventArgs: EventArgs
		{
			private bool _canceled;

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/7/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6886"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>CanceledChangedEventArgs object</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6887"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Constructor for CanceledChanged
			/// </summary>
			/// <param name="canceled"></param>
			public CanceledChangedEventArgs(bool canceled)
			{
				_canceled = canceled;
			}

			///<Developers>
			///	<Developer>John Vrooland</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/21/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6888"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6889"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// Indicates if invoice has been cancelled.
			/// </summary>
			public bool Canceled
			{
				get
				{
					return _canceled;
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		/// <summary>
		/// Pass indication of a save event, invoice has been saved.
		/// </summary>
		public class SavedChangedEventArgs: EventArgs
		{
			private bool _saved;

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/7/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6890"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>SavedChangedEventArgs object</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6891"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Constructor for this thing
			/// </summary>
			/// <param name="saved"></param>
			public SavedChangedEventArgs(bool saved)
			{
				_saved = saved;
			}

			///<Developers>
			///	<Developer>John Vrooland</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>12/2/2002</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6892"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6893"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Indicates if invoice has been saved.
			/// </summary>
			public bool Saved
			{
				get
				{
					return _saved;
				}
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/2/2002</CreationDate>
		///
		/// <summary>
		/// Indicates invoice has been confirmed.
		/// </summary>
		public class ConfirmedChangedEventArgs: EventArgs
		{
			private bool _confirmed;

			///<Developers>
			///	<Developer>Software Architectects Inc.</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/7/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6894"> 
			///		<ExpectedInput>bool</ExpectedInput>
			///		<ExpectedOutput>ConfirmedChangedEventArgs object</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6895"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///	
			/// <summary>
			/// Constructor for this thing
			/// </summary>
			/// <param name="confirmed"></param>
			public ConfirmedChangedEventArgs(bool confirmed)
			{
				_confirmed = confirmed;
			}

			///<Developers>
			///	<Developer>John Vrooland</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/21/2003</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6896"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>bool</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6897"> 
			///		<ExpectedInput>NA</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			/// <summary>
			/// Boolean to hold if the invoice has been confirmed
			/// </summary>
			public bool Confirmed
			{
				get
				{
					return _confirmed;
				}
			}
		}

		#endregion
	}
}